<?php

/**
 * This is the model class for table "PrzetargGrupa".
 *
 * The followings are the available columns in table 'PrzetargGrupa':
 * @property double $id
 * @property double $idPrzetargu
 * @property double $status
 * @property double $oferowanaCena
 * @property double $notka
 * @property string $zgloszone
 *
 * @property Przetarg        $przetarg
 * @property PrzetargFirma[] $firmy
 * @property PrzetargFirma   $owner
 * @property Firma[]         $activeCompanies
 * @property int             $tasksCount
 * @property ZlecenieFirmy[] $activeTasks
 */
class PrzetargGrupa extends CActiveRecord
{
	const TENDER_STATUS_COMPLEMENT_COMPANIES = 0;
	const TENDER_STATUS_ACCEPTATION_REQUEST = 1;
	const TENDER_STATUS_ACCEPTED = 2;
	const TENDER_STATUS_REFUSED = -1;
	const TENDER_STATUS_SUCCESS = 4;
	const TENDER_STATUS_FAILED = 5;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return PrzetargGrupa the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'PrzetargGrupa';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('idPrzetargu, oferowanaCena', 'required'),
			array('status', 'length', 'min'=>1),
			array('notka', 'length', 'max'=>2048),
			array('idPrzetargu, status', 'numerical'),
			array('oferowanaCena', 'numerical', 'min' => 1),
			array('zgloszone', 'safe'),

			array('id, idPrzetargu, status, oferowanaCena, notka, zgloszone', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'przetarg'=>array(self::BELONGS_TO, 'Przetarg', 'idPrzetargu'),
			'firmy'=>array(self::HAS_MANY, 'PrzetargFirma', 'idGrupy'),
			'owner'=>array(self::HAS_ONE, 'PrzetargFirma', 'idGrupy',
				'condition' => '"owner"."status" = ' . PrzetargFirma::STATUS_OWNER),
			'activeCompanies'=>array(self::HAS_MANY, 'Firma', array('idFirmy' => 'id'),
				'through' => 'firmy',
				'condition' => '"activeCompanies"."status" > 0 and "firmy"."status" >= ' . PrzetargFirma::STATUS_ACCEPTED),
			'tasksCount' => array(self::STAT, 'ZlecenieFirmy', 'idGrupyPrzetargowej'),
			'tasks' => array(self::HAS_MANY, 'ZlecenieFirmy', 'idGrupyPrzetargowej'),
			'activeTasks' => array(self::HAS_MANY, 'ZlecenieFirmy', 'idGrupyPrzetargowej',
				'condition' => '"activeTasks"."czasZakonczenia" is not null'),
			'failedTasks' => array(self::HAS_MANY, 'ZlecenieFirmy', 'idGrupyPrzetargowej',
				'condition' => '"failedTasks"."czasZakonczenia" is not null and "failedTasks"."czasRozpoczecia" is not null and "failedTasks"."idOdpowiedzi" is null'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('PrzetargGrupa', 'id'),
			'idPrzetargu'=>Yii::t('PrzetargGrupa', 'idZlecenia'),
			'status'=>Yii::t('PrzetargGrupa', 'status'),
			'oferowanaCena'=>Yii::t('PrzetargGrupa', 'oferowanaCena'),
			'notka'=>Yii::t('PrzetargGrupa', 'notka'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare('"id"', $this->id);
		$criteria->compare('"idPrzetargu"', $this->idZlecenia);
		$criteria->compare('"status"', $this->status);
		$criteria->compare('"oferowanaCena"', $this->oferowanaCena);
		$criteria->compare('"notka"', $this->notka);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	public static function getTeachersGroup($groupId)
	{
		$with = func_get_args();
		array_shift($with);
		return self::model()->with($with)->find(array(
			'join' => '
				LEFT JOIN "PrzetargFirma" "pf" ON "pf"."idGrupy" = "t"."id"
				LEFT JOIN "Firma" "f" ON "f"."id" = "pf"."idFirmy"
				LEFT JOIN "AdminKlasy" "ak" ON "ak"."idKlasy" = "f"."idKlasy"
			',
			'condition' => '"t"."id" = :groupId AND "ak"."idAdmina" = :teacherId AND "pf"."status" > 0',
			'params' => array(
				':groupId' => $groupId,
				':teacherId' => Yii::app()->user->id
			)
		));
	}

	public function accept()
	{
		//zaakceptuj obecną
		$this->status = self::TENDER_STATUS_ACCEPTED;
		$this->save(false);

		//oblicz ilosc zlecen
		$tasksCount = 0;
		foreach ($this->activeCompanies as $company)
		{
			$userCount = count($company->activeUsers);
			$multiplay = 1;

			if ($this->przetarg->kontrahent->idDziedziny == $company->idDziedziny || $userCount == 1)
				$multiplay = 2;

			$tasksCount += $multiplay * $userCount;
		}

		$minTasksCount = (int)ParametryGry::getValue('PRZETARG_MIN_ILOSC_ZADAN', 4);

		if($tasksCount < $minTasksCount)
			$tasksCount = $minTasksCount;

		if (empty($tasksCount))
			throw new Exception('System ustalił liczbę zleceń równą zeru. Może to
				oznaczać że w przetargu biorą udział nieaktywne firmy lub firmy składają
				się z nieaktywnych użytkowników.');

		//wylosuj zlecenia
		$tasks = Yii::app()->db->createCommand('
			select * from (
				select z."id" from "Zlecenie" z
				where z."czyPrzetarg" > 0 and z."idDziedziny" = :idDziedziny and z."poziom" = :lvl and (z."wylaczone" is null or z."wylaczone" != 1) 
				and z."kategoria" is not null and z."czas" is not null and z."cena" is not null
				ORDER BY dbms_random.value
			) where ROWNUM <= :max
		')->queryAll(true, array(
			':lvl' => $this->przetarg->poziomTrudnosci,
			':idDziedziny' => $this->przetarg->kontrahent->idDziedziny,
			':max' => $tasksCount
		));

		if (empty($tasks))
		{
			$profile = $this->przetarg->kontrahent->dziedzina->nazwa . ' - ' . $this->przetarg->poziomStr;
			throw new Exception('System nie zdołał wylosować żadnych zleceń przetargowych pasujących do profilu ' . $profile . '.');
			//	Sprawdź czy w bazie danych znadują się odpowiadające zadania z zaznaczoną flagą "Przetarg".');
		}

		$tasksCount = count($tasks);

		//dodaj zlecenia
		foreach ($tasks as $randomTask)
		{
			$task = new ZlecenieFirmy;
			$task->idZlecenia = $randomTask['id'];
			$task->idKontrahenta = $this->przetarg->idKontrahenta;
			$task->czasPodjecia = new CDbExpression('SYSDATE');
			//$task->czasPrzypisania = new CDbExpression('SYSDATE');
			$task->idGrupyPrzetargowej = $this->id;
			$task->save(false);
		}

		//wyslij wiadomość potwierdzającom do zaakceptowanej grup
		$okTitle = 'Przetarg zaakceptowany!';
		$firmy = '';
		foreach ($this->activeCompanies as $company)
			$firmy .= '<li>' . $company->nazwa . '</li>';
		$okContent = 'Nauczyciel zaakceptował przetarg ' . $this->przetarg->kontrahent->dziedzina->nazwa
			. ' - ' . $this->przetarg->poziomStr . ' dla ' . $this->przetarg->kontrahent->nazwa
			. ' na kwotę ' . number_format($this->oferowanaCena, 2, ',', ' ') . ' zł.<br/><br/>'
			. 'Następujące firmy otrzymują ' . $tasksCount . ' ' . StringUtils::numberUnitWorld($tasksCount, array('zlecenie', 'zlecenia', 'zleceń'))
			. ' do wykonania:<br/><ul>' . $firmy . '</ul>Powodzenia!';

		$this->boradcastMessage($okTitle, $okContent);

		$groups = PrzetargGrupa::model()->with('firmy.firma.activeUsers')->findAllByAttributes(array(
			'idPrzetargu' => $this->idPrzetargu,
			'status' => self::TENDER_STATUS_ACCEPTATION_REQUEST
		));

		$title = 'Przetarg odrzucony!';
		$content = 'Niestety nauczyciel odrzucił waszą ofertę przetargową ' . $this->przetarg->kontrahent->dziedzina->nazwa
			. ' - ' . $this->przetarg->poziomStr . ' dla ' . $this->przetarg->kontrahent->nazwa . '.<br/><br/>'
			. 'Wygrały następujące firmy, które otrzymują ' . $tasksCount . ' ' . StringUtils::numberUnitWorld($tasksCount, array('zlecenie', 'zlecenia', 'zleceń'))
			. ' do wykonania:<br/><ul>' . $firmy . '</ul>';

		//wyslij wiadomosc negujaca dla pozostalych grup
		foreach ($groups as $group)
		{
			$group->status = self::TENDER_STATUS_REFUSED;
			$group->save(false);

			$group->boradcastMessage($title, $content);
		}
	}

	protected function boradcastMessage($title, $content)
	{
		foreach ($this->activeCompanies as $company)
		{
			foreach ($company->activeUsers as $user)
			{
				$message = new AkcjaUzytkownika;
				$message->idAkcji = $this->przetarg->idAkcji;
				$message->idUzytkownika = $user->id;
				$message->dataWyslania = new CDbExpression('SYSDATE');
				$message->przeczytana = 0;
				$message->naglowek = $title;
				$message->tresc = $content;
				$message->save(false);
			}
		}
	}

	public function transferPayment()
	{
		$value = (float)$this->oferowanaCena / count($this->activeCompanies);

		$tenderPartialPayment = (int)ParametryGry::getValue('PRZETARG_OPLATA_CZASTKOWA', 1);
		if ($tenderPartialPayment)
		{
			$value /= (int)$this->tasksCount;
			$description = 'Cząstkowa opłata za przetarg ' . $this->przetarg->kontrahent->dziedzina->nazwa
				. ' - ' . $this->przetarg->poziomStr . ' dla ' . $this->przetarg->kontrahent->nazwa . '.';
		}
		else
		{
			$description = 'Wykonanie przetargu ' . $this->przetarg->kontrahent->dziedzina->nazwa
				. ' - ' . $this->przetarg->poziomStr . ' dla ' . $this->przetarg->kontrahent->nazwa;

			$penaltyValue = 0;
			foreach($this->failedTasks as $failedTask)
			{
				$penaltyValue += $failedTask->idZlecenia0->kara/100 * $value/(int)$this->tasksCount;
			}
			$value -= $penaltyValue;
		}

		$invoiceForCurent = null;

		foreach ($this->activeCompanies as $company)
		{
			$faktura = new Faktury;
			$faktura->czyKoszt = 0;
			$faktura->idFirmy = $company->id;
			$faktura->idKontrahenta = $this->przetarg->idKontrahenta;
			$faktura->dataWystawienia = new CDbExpression('SYSDATE');
			$faktura->opis = $description;
			$faktura->nrFaktury = 'F/' . date('Y/m/d/', time()) . rand(0, 9) . rand(0, 9) . rand(0, 9);
			$faktura->stawkaVat = ParametryGry::getFloat('PODATEK_VAT', 0.23);
			$faktura->cenaNetto = $value;
			$faktura->dataPlatnosci = new CDbExpression('SYSDATE');
			$faktura->save(false);
			if ($company->id == Yii::app()->user->id_firmy)
				$invoiceForCurent = $faktura->id;

			$przelew = new Oplaty;
			$przelew->rodzajOplaty = Oplaty::TYPE_TASK_PAYMENT;
			$przelew->idFirmy = $company->id;
			$przelew->idKontrahenta = $this->przetarg->idKontrahenta;
			$przelew->tytul = 'Opłata za fakturę ' . $faktura->nrFaktury;
			$przelew->wartosc = $faktura->cenaBrutto;
			$przelew->czas = new CDbExpression('SYSDATE');
			$przelew->save(false);

			$company->stanKonta = (float)$company->stanKonta + (float)$przelew->wartosc;
			$company->save(false);
		}

		return $invoiceForCurent;
	}

	public function payPenalty()
	{
		$value = (float)$this->oferowanaCena / count($this->activeCompanies);
		$value /= (int)$this->tasksCount;
		$penaltyValue = 0;

		foreach($this->activeTasks as $activeTask)
		{
			$penaltyValue += ((float)$activeTask->idZlecenia0->kara)/100.0 * $value;
		}

		$poziom = array(1=>'Łatwe', 2=>'Trudne', 3=>'Bardzo trudne');
		$description = 'Kara za niewykonanie przetargu '.$this->tasks[0]->idZlecenia0->dziedzina->nazwa.' - '.$poziom[$this->tasks[0]->idZlecenia0->poziom].' dla '.$this->przetarg->kontrahent->nazwa.'.';

		foreach ($this->activeCompanies as $company)
		{
			$przelew = new Oplaty;
			$przelew->rodzajOplaty = Oplaty::TYPE_TASK_PAYMENT;
			$przelew->idFirmy = $company->id;
			$przelew->idKontrahenta = $this->przetarg->idKontrahenta;
			$przelew->tytul = 'Kara za niewykonanie przetargu ' . $this->przetarg->kontrahent->nazwa.' - '.$this->przetarg->getPoziomStr();
			$przelew->wartosc = floatval((-1)*$penaltyValue);
			$przelew->czas = new CDbExpression('SYSDATE');
			$przelew->save(false);

			$company->stanKonta = (float)$company->stanKonta - (float)$penaltyValue;
			$company->save(false);

			foreach($this->tasks as $task)
			{
				$task->idOplaty = $przelew->id;
				$task->save(false);
			}
		}
	}

	public function checkStatus()
	{
		$tenderTreshold = (float)ParametryGry::getValue('PRZETARG_PROG_WYKONANIA', 1);
		$tenderFailedTasksLimit = (int)ParametryGry::getValue('PRZETARG_MAX_ZLE_ROZWIAZANYCH_ZADAN', 1) * count($this->firmy);

		$status = Yii::app()->db->createCommand('
			select
				sum(case when zf."czasZakonczenia" is not null and zf."czasRozpoczecia" is not null and zf."idOdpowiedzi" is not null then 1 else 0 end) "success",
				sum(case when zf."czasZakonczenia" is not null and zf."czasRozpoczecia" is not null and zf."idOdpowiedzi" is null then 1 else 0 end) "failed"
			from "ZlecenieFirmy" zf where zf."idGrupyPrzetargowej" = :grupaId
		')->queryRow(true, array(':grupaId'=>$this->id));

		if ((int)$status['failed'] <= $tenderFailedTasksLimit && (((int)$status['failed'] + (int)$status['success']) == $this->tasksCount))
		{
			$this->status = self::TENDER_STATUS_SUCCESS;
			$this->save(false);

			$this->boradcastMessage('Przetarg wykonany!', 'Przetarg został wykonany. Gratulacje!');

			$tenderPartialPayment = (int)ParametryGry::getValue('PRZETARG_OPLATA_CZASTKOWA', 1);
			if (!$tenderPartialPayment)
				$this->transferPayment();

			return true;
		}
		else if ((int)$status['failed'] && (int)$status['failed'] > $tenderFailedTasksLimit)
		{
			$this->status = self::TENDER_STATUS_FAILED;
			$this->save(false);

			Yii::app()->db->createCommand('
				update "ZlecenieFirmy" zf set zf."czasZakonczenia" = sysdate
				where zf."czasZakonczenia" is null and zf."idGrupyPrzetargowej" = :groupId
			')->execute(array(':groupId'=>$this->id));

			$this->payPenalty();

			$this->boradcastMessage('Przetarg niewykonany.', 'Niestety przetarg nie został wykonany.');

			return false;
		}

		return null;
	}

}
